// FormChek.js
//
// SUMMARY
//
// This is a set of JavaScript functions for validating input on 
// an HTML form.  Functions are provided to validate:
//
//      - U.S. and international phone/fax numbers
//      - U.S. ZIP codes (5 or 9 digit postal codes)
//      - U.S. Postal Codes (2 letter abbreviations for names of states)
//      - U.S. Social Security Numbers (abbreviated as SSNs)
//      - email addresses
//      - dates (entry of year, month, and day and validity of combined date)
//      - credit card numbers
//		- Pegasystem specific validations
//
//
// PEGASYSTEM FUNCTIONS:
//
// isValidNumber(input, fmt, prec, scale, req)
//				true if input is valid number based on fmt, prec, scale & req
//
// isValidTime(input, req)
//				true if input is valid time (hh:mm:ss)
//
// Supporting utility functions validate that:
//
//      - characters are Letter, Digit, or LetterOrDigit
//      - strings are a Signed, Positive, Negative, Nonpositive, or
//        Nonnegative integer
//      - strings are a Float or a SignedFloat
//      - strings are Alphabetic, Alphanumeric, or Whitespace
//      - strings contain an integer within a specified range
//
// Functions are also provided to interactively check the
// above kinds of data and prompt the user if they have
// been entered incorrectly.
//
// Other utility functions are provided to:
//
//      - remove from a string characters which are/are not 
//        in a "bag" of selected characters     
//      - reformat a string, adding delimiter characters
//      - strip whitespace/leading whitespace from a string
//      - reformat U.S. phone numbers, ZIP codes, and Social
//        Security numbers
//
//
// PEGASYSTEMS FUNCTIONS ADDED BY KAREN ZELEVINSKY AND HELEN TAYCHER	
// The parameter layer or layerName refers to the ID of the HTML DIV 
// element which contains the error image. textObj is a reference to 
// an HTML element (e.g., document.main.inputElement1).
//
// FUNCTIONS WHICH DISPLAY OR HIDE THE ERROR IMAGE
// 
// showLayer(layerName) 	
// hideLayer(layerName)
//
// FUNCTIONS USED WITH VALIDATION WHILE TYPING
//
// flagEmptyField(textObj, layer)
// flagNonZipCode(textObj, layer)
// flagNonState(textObj, layer)
// flagNonYear(textObj, layer)
// flagNonCreditCard(textObj, layer)
// flagNonUSPhoneNumber(textObj, layer)
// flagNonUS_SSN(textObj, layer)
// flagNonIntrPhoneNumber(textObj, layer)
// flagNonDay(textObj, layer)
// flagNonMonth(textObj, layer)
// flagNonDate(year,month,day, layer)
// flagNonEMail(textObj, layer)
// flagNonVisaCC(textObj, layer)
// flagNonMasterCC(textObj, layer)
// flagNonAmericanCC(textObj, layer)
// flagNonDiscoverCC(textObj, layer)
//
// FUNCTIONS USED WITH VALIDATION ON SUBMIT
//
// flagErrorsOnSubmit(thisCase, layer)	    thisCase takes any of the is... functions below 
//					    which return true if the value is valid
// isCompletedField (s)
// isPegaMonth (s)
// isPegaDay (s)
// isValidPegaPhone(s)
//
// OTHER FUNCTIONS
//
// makeUpperCase(textObj)
//
// FUNCTIONS INCLUDED FOR HISTORICAL REASONS (HAVE BEEN USED IN THE PAST)
//
// emptyField(textObj, layer)
// do_number(textObj, layer)
// do_state(textObj, layer)
// flagOnSubmit(thisCase, layer)
// isEmptyField(textObj)
// isValidZipCode(textObj)
//
//
// Many of the below functions take an optional parameter eok (for "emptyOK")
// which determines whether the empty string will return true or false.
// Default behavior is controlled by global variable defaultEmptyOK.
//
// BASIC DATA VALIDATION FUNCTIONS:
//
// isWhitespace (s)                    Check whether string s is empty or whitespace.
// isLetter (c)                        Check whether character c is an English letter 
// isDigit (c)                         Check whether character c is a digit 
// isLetterOrDigit (c)                 Check whether character c is a letter or digit.
// isInteger (s [,eok])                True if all characters in string s are numbers.
// isSignedInteger (s [,eok])          True if all characters in string s are numbers; leading + or - allowed.
// isPositiveInteger (s [,eok])        True if string s is an integer > 0.
// isNonnegativeInteger (s [,eok])     True if string s is an integer >= 0.
// isNegativeInteger (s [,eok])        True if s is an integer < 0.
// isNonpositiveInteger (s [,eok])     True if s is an integer <= 0.
// isFloat (s [,eok] [,dp])            True if string s is an unsigned floating point (real) number. (Integers also OK.)
// isSignedFloat (s [,eok] [,dp])      True if string s is a floating point number; leading + or - allowed. (Integers also OK.)
// isAlphabetic (s [,eok])             True if string s is English letters 
// isAlphanumeric (s [,eok])           True if string s is English letters and numbers only.
// isYesOrNo(s, [,eok])				   True if string s is either Y or N
// 
// isSSN (s [,eok])                    True if string s is a valid U.S. Social Security Number.
// isUSPhoneNumber (s [,eok])          True if string s is a valid U.S. Phone Number. 
// isInternationalPhoneNumber (s [,eok]) True if string s is a valid international phone number.
// isZIPCode (s [,eok])                True if string s is a valid U.S. ZIP code.
// isStateCode (s [,eok])              True if string s is a valid U.S. Postal Code
// isEmail (s [,eok])                  True if string s is a valid email address.
// isYear (s [,eok])                   True if string s is a valid Year number.
// isIntegerInRange (s, a, b [,eok])   True if string s is an integer between a and b, inclusive.
// isMonth (s [,eok])                  True if string s is a valid month between 1 and 12.
// isDay (s [,eok])                    True if string s is a valid day between 1 and 31.
// daysInFebruary (year)               Returns number of days in February of that year.
// isDate (year, month, day)           True if string arguments form a valid date.


// FUNCTIONS TO REFORMAT DATA:
//
// stripCharsInBag (s, bag)            Removes all characters in string bag from string s.
// stripCharsNotInBag (s, bag)         Removes all characters NOT in string bag from string s.
// stripWhitespace (s)                 Removes all whitespace characters from s.
// stripInitialWhitespace (s)          Removes initial (leading) whitespace characters from s.
// reformat (TARGETSTRING, STRING,     Function for inserting formatting characters or
//   INTEGER, STRING, INTEGER ... )       delimiters into TARGETSTRING.                                       
// reformatZIPCode (ZIPString)         If 9 digits, inserts separator hyphen.
// reformatSSN (SSN)                   Reformats as 123-45-6789.
// reformatUSPhone (USPhone)           Reformats as (123) 456-789.


// FUNCTIONS TO PROMPT USER:
//
// prompt (s)                          Display prompt string s in status bar.
// promptEntry (s)                     Display data entry prompt string s in status bar.
// warnEmpty (theField, s)             Notify user that required field theField is empty.
// warnInvalid (theField, s)           Notify user that contents of field theField are invalid.


// FUNCTIONS TO INTERACTIVELY CHECK FIELD CONTENTS:
//
// checkString (theField, s [,eok])    Check that theField.value is not empty or all whitespace.
// checkStateCode (theField)           Check that theField.value is a valid U.S. state code.
// checkZIPCode (theField [,eok])      Check that theField.value is a valid ZIP code.
// checkUSPhone (theField [,eok])      Check that theField.value is a valid US Phone.
// checkInternationalPhone (theField [,eok])  Check that theField.value is a valid International Phone.
// checkEmail (theField [,eok])        Check that theField.value is a valid Email.
// checkSSN (theField [,eok])          Check that theField.value is a valid SSN.
// checkYear (theField [,eok])         Check that theField.value is a valid Year.
// checkMonth (theField [,eok])        Check that theField.value is a valid Month.
// checkDay (theField [,eok])          Check that theField.value is a valid Day.
// checkDate (yearField, monthField, dayField, labelString, OKtoOmitDay)
//                                     Check that field values form a valid date.
// getRadioButtonValue (radio)         Get checked value from radio button.
// checkCreditCard (radio, theField)   Validate credit card info.


// CREDIT CARD DATA VALIDATION FUNCTIONS
// 
// isCreditCard (st)              True if credit card number passes the Luhn Mod-10 test.
// isVisa (cc)                    True if string cc is a valid VISA number.
// isMasterCard (cc)              True if string cc is a valid MasterCard number.
// isAmericanExpress (cc)         True if string cc is a valid American Express number.
// isDinersClub (cc)              True if string cc is a valid Diner's Club number.
// isCarteBlanche (cc)            True if string cc is a valid Carte Blanche number.
// isDiscover (cc)                True if string cc is a valid Discover card number.
// isEnRoute (cc)                 True if string cc is a valid enRoute card number.
// isJCB (cc)                     True if string cc is a valid JCB card number.
// isAnyCard (cc)                 True if string cc is a valid card number for any of the accepted types.
// isCardMatch (Type, Number)     True if Number is valid for credic card of type Type.
//
// Other stub functions are retained for backward compatibility with LivePayment code.
// See comments below for details.
//
// Performance hint: when you deploy this file on your website, strip out the
// comment lines from the source code as well as any of the functions which
// you don't need.  This will give you a smaller .js file and achieve faster
// downloads.
//
// 18 Feb 97 created Eric Krock
//
// (c) 1997 Netscape Communications Corporation



// VARIABLE DECLARATIONS

var digits = "0123456789";

var lowercaseLetters = "abcdefghijklmnopqrstuvwxyz"

var uppercaseLetters = "ABCDEFGHIJKLMNOPQRSTUVWXYZ"

// whitespace characters
var whitespace = " \t\n\r";


// decimal point character differs by language and culture
var decimalPointDelimiter = "."


// non-digit characters which are allowed in phone numbers
var phoneNumberDelimiters = "()- ";


// characters which are allowed in US phone numbers
var validUSPhoneChars = digits + phoneNumberDelimiters;


// characters which are allowed in international phone numbers
// (a leading + is OK)
var validWorldPhoneChars = digits + phoneNumberDelimiters + "+";


// non-digit characters which are allowed in 
// Social Security Numbers
var SSNDelimiters = "- ";



// characters which are allowed in Social Security Numbers
var validSSNChars = digits + SSNDelimiters;



// U.S. Social Security Numbers have 9 digits.
// They are formatted as 123-45-6789.
var digitsInSocialSecurityNumber = 9;



// U.S. phone numbers have 10 digits.
// They are formatted as 123 456 7890 or (123) 456-7890.
var digitsInUSPhoneNumber = 10;



// non-digit characters which are allowed in ZIP Codes
var ZIPCodeDelimiters = "-";



// our preferred delimiter for reformatting ZIP Codes
var ZIPCodeDelimeter = "-"


// characters which are allowed in Social Security Numbers
var validZIPCodeChars = digits + ZIPCodeDelimiters



// U.S. ZIP codes have 5 or 9 digits.
// They are formatted as 12345 or 12345-6789.
var digitsInZIPCode1 = 5
var digitsInZIPCode2 = 9


// non-digit characters which are allowed in credit card numbers
var creditCardDelimiters = " "


// CONSTANT STRING DECLARATIONS
// (grouped for ease of translation and localization)

// m is an abbreviation for "missing"

var mPrefix = "You did not enter a value into the "
var mSuffix = " field. This is a required field. Please enter it now."

// s is an abbreviation for "string"

var sUSLastName = "Last Name"
var sUSFirstName = "First Name"
var sWorldLastName = "Family Name"
var sWorldFirstName = "Given Name"
var sTitle = "Title"
var sCompanyName = "Company Name"
var sUSAddress = "Street Address"
var sWorldAddress = "Address"
var sCity = "City"
var sStateCode = "State Code"
var sWorldState = "State, Province, or Prefecture"
var sCountry = "Country"
var sZIPCode = "ZIP Code"
var sWorldPostalCode = "Postal Code"
var sPhone = "Phone Number"
var sFax = "Fax Number"
var sDateOfBirth = "Date of Birth"
var sExpirationDate = "Expiration Date"
var sEmail = "Email"
var sSSN = "Social Security Number"
var sCreditCardNumber = "Credit Card Number"
var sOtherInfo = "Other Information"




// i is an abbreviation for "invalid"

var iStateCode = "This field must be a valid two character U.S. state abbreviation (like CA for California). Please reenter it now."
var iZIPCode = "This field must be a 5 or 9 digit U.S. ZIP Code (like 94043). Please reenter it now."
var iUSPhone = "This field must be a 10 digit U.S. phone number (like 415 555 1212). Please reenter it now."
var iWorldPhone = "This field must be a valid international phone number. Please reenter it now."
var iSSN = "This field must be a 9 digit U.S. social security number (like 123 45 6789). Please reenter it now."
var iEmail = "This field must be a valid email address (like foo@bar.com). Please reenter it now."
var iCreditCardPrefix = "This is not a valid "
var iCreditCardSuffix = " credit card number. (Click the link on this form to see a list of sample numbers.) Please reenter it now."
var iDay = "This field must be a day number between 1 and 31.  Please reenter it now."
var iMonth = "This field must be a month number between 1 and 12.  Please reenter it now."
var iYear = "This field must be a 2 or 4 digit year number.  Please reenter it now."
var iDatePrefix = "The Day, Month, and Year for "
var iDateSuffix = " do not form a valid date.  Please reenter them now."



// p is an abbreviation for "prompt"

var pEntryPrompt = "Please enter a "
var pStateCode = "2 character code (like CA)."
var pZIPCode = "5 or 9 digit U.S. ZIP Code (like 94043)."
var pUSPhone = "10 digit U.S. phone number (like 415 555 1212)."
var pWorldPhone = "international phone number."
var pSSN = "9 digit U.S. social security number (like 123 45 6789)."
var pEmail = "valid email address (like foo@bar.com)."
var pCreditCard = "valid credit card number."
var pDay = "day number between 1 and 31."
var pMonth = "month number between 1 and 12."
var pYear = "2 or 4 digit year number."


// Global variable defaultEmptyOK defines default return value 
// for many functions when they are passed the empty string. 
// By default, they will return defaultEmptyOK.
//
// defaultEmptyOK is false, which means that by default, 
// these functions will do "strict" validation.  Function
// isInteger, for example, will only return true if it is
// passed a string containing an integer; if it is passed
// the empty string, it will return false.
//
// You can change this default behavior globally (for all 
// functions which use defaultEmptyOK) by changing the value
// of defaultEmptyOK.
//
// Most of these functions have an optional argument emptyOK
// which allows you to override the default behavior for 
// the duration of a function call.
//
// This functionality is useful because it is possible to
// say "if the user puts anything in this field, it must
// be an integer (or a phone number, or a string, etc.), 
// but it's OK to leave the field empty too."
// This is the case for fields which are optional but which
// must have a certain kind of content if filled in.

var defaultEmptyOK = false




// Attempting to make this library run on Navigator 2.0,
// so I'm supplying this array creation routine as per
// JavaScript 1.0 documentation.  If you're using 
// Navigator 3.0 or later, you don't need to do this;
// you can use the Array constructor instead.

function makeArray(n) {
//*** BUG: If I put this line in, I get two error messages:
//(1) Window.length can't be set by assignment
//(2) daysInMonth has no property indexed by 4
//If I leave it out, the code works fine.
//   this.length = n;
   for (var i = 1; i <= n; i++) {
      this[i] = 0
   } 
   return this
}



var daysInMonth = makeArray(12);
daysInMonth[1] = 31;
daysInMonth[2] = 29;   // must programmatically check this
daysInMonth[3] = 31;
daysInMonth[4] = 30;
daysInMonth[5] = 31;
daysInMonth[6] = 30;
daysInMonth[7] = 31;
daysInMonth[8] = 31;
daysInMonth[9] = 30;
daysInMonth[10] = 31;
daysInMonth[11] = 30;
daysInMonth[12] = 31;




// Valid U.S. Postal Codes for states, territories, armed forces, etc.
// See http://www.usps.gov/ncsc/lookups/abbr_state.txt.

var USStateCodeDelimiter = "|";
var USStateCodes = "AL|AK|AS|AZ|AR|CA|CO|CT|DE|DC|FM|FL|GA|GU|HI|ID|IL|IN|IA|KS|KY|LA|ME|MH|MD|MA|MI|MN|MS|MO|MT|NE|NV|NH|NJ|NM|NY|NC|ND|MP|OH|OK|OR|PW|PA|PR|RI|SC|SD|TN|TX|UT|VT|VI|VA|WA|WV|WI|WY|AE|AA|AE|AE|AP"




// Check whether string s is empty.

function isEmpty(s)
{
   return ((s == null) || (s.length == 0));
}



// Returns true if string s is empty or 
// whitespace characters only.

function isWhitespace (s)

{   var i;

    // Is s empty?
    if (isEmpty(s)) return true;

    // Search through string's characters one by one
    // until we find a non-whitespace character.
    // When we do, return false; if we don't, return true.

    for (i = 0; i < s.length; i++)
    {   
        // Check that current character isn't whitespace.
        var c = s.charAt(i);

        if (whitespace.indexOf(c) == -1) return false;
    }

    // All characters are whitespace.
    return true;
}



// Removes all characters which appear in string bag from string s.

function stripCharsInBag (s, bag)

{   var i;
    var returnString = "";

    // Search through string's characters one by one.
    // If character is not in bag, append to returnString.

    for (i = 0; i < s.length; i++)
    {   
        // Check that current character isn't whitespace.
        var c = s.charAt(i);
        if (bag.indexOf(c) == -1) returnString += c;
    }

    return returnString;
}



// Removes all characters which do NOT appear in string bag 
// from string s.

function stripCharsNotInBag (s, bag)

{   var i;
    var returnString = "";

    // Search through string's characters one by one.
    // If character is in bag, append to returnString.

    for (i = 0; i < s.length; i++)
    {   
        // Check that current character isn't whitespace.
        var c = s.charAt(i);
        if (bag.indexOf(c) != -1) returnString += c;
    }

    return returnString;
}



// Removes all whitespace characters from s.
// Global variable whitespace (see above)
// defines which characters are considered whitespace.

function stripWhitespace (s)

{   return stripCharsInBag (s, whitespace)
}




// WORKAROUND FUNCTION FOR NAVIGATOR 2.0.2 COMPATIBILITY.
//
// The below function *should* be unnecessary.  In general,
// avoid using it.  Use the standard method indexOf instead.
//
// However, because of an apparent bug in indexOf on 
// Navigator 2.0.2, the below loop does not work as the
// body of stripInitialWhitespace:
//
// while ((i < s.length) && (whitespace.indexOf(s.charAt(i)) != -1))
//   i++;
//
// ... so we provide this workaround function charInString
// instead.
//
// charInString (CHARACTER c, STRING s)
//
// Returns true if single character c (actually a string)
// is contained within string s.

function charInString (c, s)
{   for (i = 0; i < s.length; i++)
    {   if (s.charAt(i) == c) return true;
    }
    return false
}



// Removes initial (leading) whitespace characters from s.
// Global variable whitespace (see above)
// defines which characters are considered whitespace.

function stripInitialWhitespace (s)

{   var i = 0;

    while ((i < s.length) && charInString (s.charAt(i), whitespace))
       i++;
    
    return s.substring (i, s.length);
}







// Returns true if character c is an English letter 
// (A .. Z, a..z).
//
// NOTE: Need i18n version to support European characters.
// This could be tricky due to different character
// sets and orderings for various languages and platforms.

function isLetter (c)
{   return ( ((c >= "a") && (c <= "z")) || ((c >= "A") && (c <= "Z")) )
}



// Returns true if character c is a digit 
// (0 .. 9).

function isDigit (c)
{   return ((c >= "0") && (c <= "9"))
}



// Returns true if character c is a letter or digit.

function isLetterOrDigit (c)
{   return (isLetter(c) || isDigit(c))
}



// isInteger (STRING s [, BOOLEAN emptyOK])
// 
// Returns true if all characters in string s are numbers.
//
// Accepts non-signed integers only. Does not accept floating 
// point, exponential notation, etc.
//
// We don't use parseInt because that would accept a string
// with trailing non-numeric characters.
//
// By default, returns defaultEmptyOK if s is empty.
// There is an optional second argument called emptyOK.
// emptyOK is used to override for a single function call
//      the default behavior which is specified globally by
//      defaultEmptyOK.
// If emptyOK is false (or any value other than true), 
//      the function will return false if s is empty.
// If emptyOK is true, the function will return true if s is empty.
//
// EXAMPLE FUNCTION CALL:     RESULT:
// isInteger ("5")            true 
// isInteger ("")             defaultEmptyOK
// isInteger ("-5")           false
// isInteger ("", true)       true
// isInteger ("", false)      false
// isInteger ("5", false)     true

function isInteger (s)

{   var i;

    if (isEmpty(s)) 
       if (isInteger.arguments.length == 1) return defaultEmptyOK;
       else return (isInteger.arguments[1] == true);

    // Search through string's characters one by one
    // until we find a non-numeric character.
    // When we do, return false; if we don't, return true.

    for (i = 0; i < s.length; i++)
    {   
        // Check that current character is number.
        var c = s.charAt(i);

        if (!isDigit(c)) return false;
    }

    // All characters are numbers.
    return true;
}







// isSignedInteger (STRING s [, BOOLEAN emptyOK])
// 
// Returns true if all characters are numbers; 
// first character is allowed to be + or - as well.
//
// Does not accept floating point, exponential notation, etc.
//
// We don't use parseInt because that would accept a string
// with trailing non-numeric characters.
//
// For explanation of optional argument emptyOK,
// see comments of function isInteger.
//
// EXAMPLE FUNCTION CALL:          RESULT:
// isSignedInteger ("5")           true 
// isSignedInteger ("")            defaultEmptyOK
// isSignedInteger ("-5")          true
// isSignedInteger ("+5")          true
// isSignedInteger ("", false)     false
// isSignedInteger ("", true)      true

function isSignedInteger (s)

{
   if (isEmpty(s)) 
       if (isSignedInteger.arguments.length == 1) return defaultEmptyOK;
       else return (isSignedInteger.arguments[1] == true);

    else {
        var startPos = 0;
        var secondArg = defaultEmptyOK;

        if (isSignedInteger.arguments.length > 1)
            secondArg = isSignedInteger.arguments[1];

        // skip leading + or -
        if ( (s.charAt(0) == "-") || (s.charAt(0) == "+") )
           startPos = 1;    
        return (isInteger(s.substring(startPos, s.length), secondArg))
    }
}




// isPositiveInteger (STRING s [, BOOLEAN emptyOK])
// 
// Returns true if string s is an integer > 0.
//
// For explanation of optional argument emptyOK,
// see comments of function isInteger.

function isPositiveInteger (s)
{   var secondArg = defaultEmptyOK;

    if (isPositiveInteger.arguments.length > 1)
        secondArg = isPositiveInteger.arguments[1];

    // The next line is a bit byzantine.  What it means is:
    // a) s must be a signed integer, AND
    // b) one of the following must be true:
    //    i)  s is empty and we are supposed to return true for
    //        empty strings
    //    ii) this is a positive, not negative, number

    return (isSignedInteger(s, secondArg)
         && ( (isEmpty(s) && secondArg)  || (parseInt (s) > 0) ) );
}






// isNonnegativeInteger (STRING s [, BOOLEAN emptyOK])
// 
// Returns true if string s is an integer >= 0.
//
// For explanation of optional argument emptyOK,
// see comments of function isInteger.

function isNonnegativeInteger (s)
{   var secondArg = defaultEmptyOK;

    if (isNonnegativeInteger.arguments.length > 1)
        secondArg = isNonnegativeInteger.arguments[1];

    // The next line is a bit byzantine.  What it means is:
    // a) s must be a signed integer, AND
    // b) one of the following must be true:
    //    i)  s is empty and we are supposed to return true for
    //        empty strings
    //    ii) this is a number >= 0

    return (isSignedInteger(s, secondArg)
         && ( (isEmpty(s) && secondArg)  || (parseInt (s) >= 0) ) );
}






// isNegativeInteger (STRING s [, BOOLEAN emptyOK])
// 
// Returns true if string s is an integer < 0.
//
// For explanation of optional argument emptyOK,
// see comments of function isInteger.

function isNegativeInteger (s)
{   var secondArg = defaultEmptyOK;

    if (isNegativeInteger.arguments.length > 1)
        secondArg = isNegativeInteger.arguments[1];

    // The next line is a bit byzantine.  What it means is:
    // a) s must be a signed integer, AND
    // b) one of the following must be true:
    //    i)  s is empty and we are supposed to return true for
    //        empty strings
    //    ii) this is a negative, not positive, number

    return (isSignedInteger(s, secondArg)
         && ( (isEmpty(s) && secondArg)  || (parseInt (s) < 0) ) );
}






// isNonpositiveInteger (STRING s [, BOOLEAN emptyOK])
// 
// Returns true if string s is an integer <= 0.
//
// For explanation of optional argument emptyOK,
// see comments of function isInteger.

function isNonpositiveInteger (s)
{   var secondArg = defaultEmptyOK;

    if (isNonpositiveInteger.arguments.length > 1)
        secondArg = isNonpositiveInteger.arguments[1];

    // The next line is a bit byzantine.  What it means is:
    // a) s must be a signed integer, AND
    // b) one of the following must be true:
    //    i)  s is empty and we are supposed to return true for
    //        empty strings
    //    ii) this is a number <= 0

    return (isSignedInteger(s, secondArg)
         && ( (isEmpty(s) && secondArg)  || (parseInt (s) <= 0) ) );
}





// isFloat (STRING s [, BOOLEAN emptyOK] [, STRING decimalPoint])
// 
// True if string s is an unsigned floating point (real) number. 
//
// Also returns true for unsigned integers. If you wish
// to distinguish between integers and floating point numbers,
// first call isInteger, then call isFloat.
//
// Does not accept exponential notation.
//
// For explanation of optional argument emptyOK,
// see comments of function isInteger.

function isFloat (s)

{   var i;
    var seenDecimalPoint = false;
	var dp = decimalPointDelimiter;

	if (isFloat.arguments.length > 2)
		dp = isFloat.arguments[2];

    if (isEmpty(s)) 
       if (isFloat.arguments.length == 1) return defaultEmptyOK;
       else return (isFloat.arguments[1] == true);

    if (s == dp) return false;

    // Search through string's characters one by one
    // until we find a non-numeric character.
    // When we do, return false; if we don't, return true.

    for (i = 0; i < s.length; i++)
    {   
        // Check that current character is number.
        var c = s.charAt(i);

        if ((c == dp) && !seenDecimalPoint) seenDecimalPoint = true;
        else if (!isDigit(c)) return false;
    }

    // All characters are numbers.
    return true;
}







// isSignedFloat (STRING s [, BOOLEAN emptyOK] [, STRING decimalPoint])
// 
// True if string s is a signed or unsigned floating point 
// (real) number. First character is allowed to be + or -.
//
// Also returns true for unsigned integers. If you wish
// to distinguish between integers and floating point numbers,
// first call isSignedInteger, then call isSignedFloat.
//
// Does not accept exponential notation.
//
// For explanation of optional argument emptyOK,
// see comments of function isInteger.

function isSignedFloat (s)

{   if (isEmpty(s)) 
       if (isSignedFloat.arguments.length == 1) return defaultEmptyOK;
       else return (isSignedFloat.arguments[1] == true);

    else {
        var startPos = 0;
        var secondArg = defaultEmptyOK;
		var thirdArg = decimalPointDelimiter

        if (isSignedFloat.arguments.length > 1)
            secondArg = isSignedFloat.arguments[1];

        if (isSignedFloat.arguments.length > 2)
            thirdArg = isSignedFloat.arguments[2];

        // skip leading + or -
        if ( (s.charAt(0) == "-") || (s.charAt(0) == "+") )
           startPos = 1;    
        return (isFloat(s.substring(startPos, s.length), secondArg, thirdArg))
    }
}




// isAlphabetic (STRING s [, BOOLEAN emptyOK])
// 
// Returns true if string s is English letters 
// (A .. Z, a..z) only.
//
// For explanation of optional argument emptyOK,
// see comments of function isInteger.
//
// NOTE: Need i18n version to support European characters.
// This could be tricky due to different character
// sets and orderings for various languages and platforms.

function isAlphabetic (s)

{   var i;

    if (isEmpty(s)) 
       if (isAlphabetic.arguments.length == 1) return defaultEmptyOK;
       else return (isAlphabetic.arguments[1] == true);

    // Search through string's characters one by one
    // until we find a non-alphabetic character.
    // When we do, return false; if we don't, return true.

    for (i = 0; i < s.length; i++)
    {   
        // Check that current character is letter.
        var c = s.charAt(i);

        if (!isLetter(c))
        return false;
    }

    // All characters are letters.
    return true;
}




// isAlphanumeric (STRING s [, BOOLEAN emptyOK])
// 
// Returns true if string s is English letters 
// (A .. Z, a..z) and numbers only.
//
// For explanation of optional argument emptyOK,
// see comments of function isInteger.
//
// NOTE: Need i18n version to support European characters.
// This could be tricky due to different character
// sets and orderings for various languages and platforms.

function isAlphanumeric (s)

{   var i;

    if (isEmpty(s)) 
       if (isAlphanumeric.arguments.length == 1) return defaultEmptyOK;
       else return (isAlphanumeric.arguments[1] == true);

    // Search through string's characters one by one
    // until we find a non-alphanumeric character.
    // When we do, return false; if we don't, return true.

    for (i = 0; i < s.length; i++)
    {   
        // Check that current character is number or letter.
        var c = s.charAt(i);

        if (! (isLetter(c) || isDigit(c) ) )
        return false;
    }

    // All characters are numbers or letters.
    return true;
}




// isYesOrNo (STRING s [, BOOLEAN emptyOK])
// 
// Returns true if string s is either Y, y, N or n
//
// For explanation of optional argument emptyOK,
// see comments of function isInteger.
//
function isYesOrNo(s)

{
    var i;

    if (isEmpty(s)) 
       if (isYesOrNo.arguments.length == 1) return defaultEmptyOK;
       else return (isYesOrNo.arguments[1] == true);

    // Length of string must be 1
	if (s.length > 1)
	   return false;

    // check for Y, y, N, n
	if (s.toUpperCase() != "Y" && s.toUpperCase() != "N")
	    return false;

    // success
    return true;
}


// reformat (TARGETSTRING, STRING, INTEGER, STRING, INTEGER ... )       
//
// Handy function for arbitrarily inserting formatting characters
// or delimiters of various kinds within TARGETSTRING.
//
// reformat takes one named argument, a string s, and any number
// of other arguments.  The other arguments must be integers or
// strings.  These other arguments specify how string s is to be
// reformatted and how and where other strings are to be inserted
// into it.
//
// reformat processes the other arguments in order one by one.
// * If the argument is an integer, reformat appends that number 
//   of sequential characters from s to the resultString.
// * If the argument is a string, reformat appends the string
//   to the resultString.
//
// NOTE: The first argument after TARGETSTRING must be a string.
// (It can be empty.)  The second argument must be an integer.
// Thereafter, integers and strings must alternate.  This is to
// provide backward compatibility to Navigator 2.0.2 JavaScript
// by avoiding use of the typeof operator.
//
// It is the caller's responsibility to make sure that we do not
// try to copy more characters from s than s.length.
//
// EXAMPLES:
//
// * To reformat a 10-digit U.S. phone number from "1234567890"
//   to "(123) 456-7890" make this function call:
//   reformat("1234567890", "(", 3, ") ", 3, "-", 4)
//
// * To reformat a 9-digit U.S. Social Security number from
//   "123456789" to "123-45-6789" make this function call:
//   reformat("123456789", "", 3, "-", 2, "-", 4)
//
// HINT:
//
// If you have a string which is already delimited in one way
// (example: a phone number delimited with spaces as "123 456 7890")
// and you want to delimit it in another way using function reformat,
// call function stripCharsNotInBag to remove the unwanted 
// characters, THEN call function reformat to delimit as desired.
//
// EXAMPLE:
//
// reformat (stripCharsNotInBag ("123 456 7890", digits),
//           "(", 3, ") ", 3, "-", 4)

function reformat (s)

{   var arg;
    var sPos = 0;
    var resultString = "";

    for (var i = 1; i < reformat.arguments.length; i++) {
       arg = reformat.arguments[i];
       if (i % 2 == 1) resultString += arg;
       else {
           resultString += s.substring(sPos, sPos + arg);
           sPos += arg;
       }
    }
    return resultString;
}




// isSSN (STRING s [, BOOLEAN emptyOK])
// 
// isSSN returns true if string s is a valid U.S. Social
// Security Number.  Must be 9 digits.
//
// NOTE: Strip out any delimiters (spaces, hyphens, etc.)
// from string s before calling this function.
//
// For explanation of optional argument emptyOK,
// see comments of function isInteger.

function isSSN (s)
{   if (isEmpty(s)) 
       if (isSSN.arguments.length == 1) return defaultEmptyOK;
       else return (isSSN.arguments[1] == true);
    return (isInteger(s) && s.length == digitsInSocialSecurityNumber)
}




// isUSPhoneNumber (STRING s [, BOOLEAN emptyOK])
// 
// isUSPhoneNumber returns true if string s is a valid U.S. Phone
// Number.  Must be 10 digits.
//
// NOTE: Strip out any delimiters (spaces, hyphens, parentheses, etc.)
// from string s before calling this function.
//
// For explanation of optional argument emptyOK,
// see comments of function isInteger.

function isUSPhoneNumber (s)
{   if (isEmpty(s)) 
       if (isUSPhoneNumber.arguments.length == 1) return defaultEmptyOK;
       else return (isUSPhoneNumber.arguments[1] == true);
    return (isInteger(s) && s.length == digitsInUSPhoneNumber)
}




// isInternationalPhoneNumber (STRING s [, BOOLEAN emptyOK])
// 
// isInternationalPhoneNumber returns true if string s is a valid 
// international phone number.  Must be digits only; any length OK.
// May be prefixed by + character.
//
// NOTE: A phone number of all zeros would not be accepted.
// I don't think that is a valid phone number anyway.
//
// NOTE: Strip out any delimiters (spaces, hyphens, parentheses, etc.)
// from string s before calling this function.  You may leave in 
// leading + character if you wish.
//
// For explanation of optional argument emptyOK,
// see comments of function isInteger.

function isInternationalPhoneNumber (s)
{   if (isEmpty(s)) 
       if (isInternationalPhoneNumber.arguments.length == 1) return defaultEmptyOK;
       else return (isInternationalPhoneNumber.arguments[1] == true);
    return (isPositiveInteger(s))
}




// isZIPCode (STRING s [, BOOLEAN emptyOK])
// 
// isZIPCode returns true if string s is a valid 
// U.S. ZIP code.  Must be 5 or 9 digits only.
//
// NOTE: Strip out any delimiters (spaces, hyphens, etc.)
// from string s before calling this function.  
//
// For explanation of optional argument emptyOK,
// see comments of function isInteger.

function isZIPCode (s)
{  if (isEmpty(s)) 
       if (isZIPCode.arguments.length == 1) return defaultEmptyOK;
       else return (isZIPCode.arguments[1] == true);
   return (isInteger(s) && 
            ((s.length == digitsInZIPCode1) ||
             (s.length == digitsInZIPCode2)))
}





// isStateCode (STRING s [, BOOLEAN emptyOK])
// 
// Return true if s is a valid U.S. Postal Code 
// (abbreviation for state).
//
// For explanation of optional argument emptyOK,
// see comments of function isInteger.

function isStateCode(s)
{   if (isEmpty(s)) 
       if (isStateCode.arguments.length == 1) return defaultEmptyOK;
       else return (isStateCode.arguments[1] == true);
    return ( (USStateCodes.indexOf(s) != -1) &&
             (s.indexOf(USStateCodeDelimiter) == -1) )
}




// isEmail (STRING s [, BOOLEAN emptyOK])
// 
// Email address must be of form a@b.c -- in other words:
// * there must be at least one character before the @
// * there must be at least one character before and after the .
// * the characters @ and . are both required
//
// For explanation of optional argument emptyOK,
// see comments of function isInteger.

function isEmail (s)
{   if (isEmpty(s)) 
       if (isEmail.arguments.length == 1) return defaultEmptyOK;
       else return (isEmail.arguments[1] == true);
   
    // is s whitespace?
    if (isWhitespace(s)) return false;
    
    // there must be >= 1 character before @, so we
    // start looking at character position 1 
    // (i.e. second character)
    var i = 1;
    var sLength = s.length;

    // look for @
    while ((i < sLength) && (s.charAt(i) != "@"))
    { i++
    }

    if ((i >= sLength) || (s.charAt(i) != "@")) return false;
    else i += 2;

    // look for .
    while ((i < sLength) && (s.charAt(i) != "."))
    { i++
    }

    // there must be at least one character after the .
    if ((i >= sLength - 1) || (s.charAt(i) != ".")) return false;
    else return true;
}





// isYear (STRING s [, BOOLEAN emptyOK])
// 
// isYear returns true if string s is a valid 
// Year number.  Must be 2 or 4 digits only.
// 
// For Year 2000 compliance, you are advised
// to use 4-digit year numbers everywhere.
//
// And yes, this function is not Year 10000 compliant, but 
// because I am giving you 8003 years of advance notice,
// I don't feel very guilty about this ...
//
// For B.C. compliance, write your own function. ;->
//
// For explanation of optional argument emptyOK,
// see comments of function isInteger.

function isYear (s)
{   if (isEmpty(s)) 
       if (isYear.arguments.length == 1) return defaultEmptyOK;
       else return (isYear.arguments[1] == true);
    if (!isNonnegativeInteger(s)) return false;
    return ((s.length == 2) || (s.length == 4));
}



// isIntegerInRange (STRING s, INTEGER a, INTEGER b [, BOOLEAN emptyOK])
// 
// isIntegerInRange returns true if string s is an integer 
// within the range of integer arguments a and b, inclusive.
// 
// For explanation of optional argument emptyOK,
// see comments of function isInteger.


function isIntegerInRange (s, a, b)
{   if (isEmpty(s)) 
       if (isIntegerInRange.arguments.length == 1) return defaultEmptyOK;
       else return (isIntegerInRange.arguments[1] == true);

    // Catch non-integer strings to avoid creating a NaN below,
    // which isn't available on JavaScript 1.0 for Windows.
    if (!isInteger(s, false)) return false;

    // Now, explicitly change the type to integer via parseInt
    // so that the comparison code below will work both on 
    // JavaScript 1.2 (which typechecks in equality comparisons)
    // and JavaScript 1.1 and before (which doesn't).
    var num = parseInt (s);
    return ((num >= a) && (num <= b));
}



// isMonth (STRING s [, BOOLEAN emptyOK])
// 
// isMonth returns true if string s is a valid 
// month number between 1 and 12.
//
// For explanation of optional argument emptyOK,
// see comments of function isInteger.

function isMonth (s)
{   if (isEmpty(s)) 
       if (isMonth.arguments.length == 1) return defaultEmptyOK;
       else return (isMonth.arguments[1] == true);
    return isIntegerInRange (s, 1, 12);
}



// isDay (STRING s [, BOOLEAN emptyOK])
// 
// isDay returns true if string s is a valid 
// day number between 1 and 31.
// 
// For explanation of optional argument emptyOK,
// see comments of function isInteger.

function isDay (s)
{   if (isEmpty(s)) 
       if (isDay.arguments.length == 1) return defaultEmptyOK;
       else return (isDay.arguments[1] == true);   
    return isIntegerInRange (s, 1, 31);
}



// daysInFebruary (INTEGER year)
// 
// Given integer argument year,
// returns number of days in February of that year.

function daysInFebruary (year)
{   // February has 29 days in any year evenly divisible by four,
    // EXCEPT for centurial years which are not also divisible by 400.
    return (  ((year % 4 == 0) && ( (!(year % 100 == 0)) || (year % 400 == 0) ) ) ? 29 : 28 );
}



// isDate (STRING year, STRING month, STRING day)
//
// isDate returns true if string arguments year, month, and day 
// form a valid date.
// 

function isDate (year, month, day)
{   // catch invalid years (not 2- or 4-digit) and invalid months and days.
    if (! (isYear(year, false) && isMonth(month, false) && isDay(day, false))) return false;

    // Explicitly change type to integer to make code work in both
    // JavaScript 1.1 and JavaScript 1.2.
    var intYear = parseInt(year);
    var intMonth = parseInt(month);
    var intDay = parseInt(day);

    // catch invalid days, except for February
    if (intDay > daysInMonth[intMonth]) return false; 

    if ((intMonth == 2) && (intDay > daysInFebruary(intYear))) return false;

    return true;
}




/* FUNCTIONS TO NOTIFY USER OF INPUT REQUIREMENTS OR MISTAKES. */


// Display prompt string s in status bar.

function prompt (s)
{   window.status = s
}



// Display data entry prompt string s in status bar.

function promptEntry (s)
{   window.status = pEntryPrompt + s
}




// Notify user that required field theField is empty.
// String s describes expected contents of theField.value.
// Put focus in theField and return false.

function warnEmpty (theField, s)
{   theField.focus()
    alert(mPrefix + s + mSuffix)
    return false
}



// Notify user that contents of field theField are invalid.
// String s describes expected contents of theField.value.
// Put select theField, pu focus in it, and return false.

function warnInvalid (theField, s)
{   theField.focus()
    theField.select()
    alert(s)
    return false
}




/* FUNCTIONS TO INTERACTIVELY CHECK VARIOUS FIELDS. */

// checkString (TEXTFIELD theField, STRING s, [, BOOLEAN emptyOK==false])
//
// Check that string theField.value is not all whitespace.
//
// For explanation of optional argument emptyOK,
// see comments of function isInteger.

function checkString (theField, s, emptyOK)
{   // Next line is needed on NN3 to avoid "undefined is not a number" error
    // in equality comparison below.
    if (checkString.arguments.length == 2) emptyOK = defaultEmptyOK;
    if ((emptyOK == true) && (isEmpty(theField.value))) return true;
    if (isWhitespace(theField.value)) 
       return warnEmpty (theField, s);
    else return true;
}



// checkStateCode (TEXTFIELD theField [, BOOLEAN emptyOK==false])
//
// Check that string theField.value is a valid U.S. state code.
//
// For explanation of optional argument emptyOK,
// see comments of function isInteger.

function checkStateCode (theField, emptyOK)
{   if (checkStateCode.arguments.length == 1) emptyOK = defaultEmptyOK;
    if ((emptyOK == true) && (isEmpty(theField.value))) return true;
    else
    {  theField.value = theField.value.toUpperCase();
       if (!isStateCode(theField.value, false)) 
          return warnInvalid (theField, iStateCode);
       else return true;
    }
}



// takes ZIPString, a string of 5 or 9 digits;
// if 9 digits, inserts separator hyphen

function reformatZIPCode (ZIPString)
{   if (ZIPString.length == 5) return ZIPString;
    else return (reformat (ZIPString, "", 5, "-", 4));
}




// checkZIPCode (TEXTFIELD theField [, BOOLEAN emptyOK==false])
//
// Check that string theField.value is a valid ZIP code.
//
// For explanation of optional argument emptyOK,
// see comments of function isInteger.

function checkZIPCode (theField, emptyOK)
{   if (checkZIPCode.arguments.length == 1) emptyOK = defaultEmptyOK;
    if ((emptyOK == true) && (isEmpty(theField.value))) return true;
    else
    { var normalizedZIP = stripCharsInBag(theField.value, ZIPCodeDelimiters)
      if (!isZIPCode(normalizedZIP, false)) 
         return warnInvalid (theField, iZIPCode);
      else 
      {  // if you don't want to insert a hyphen, comment next line out
         theField.value = reformatZIPCode(normalizedZIP)
         return true;
      }
    }
}



// takes USPhone, a string of 10 digits
// and reformats as (123) 456-789

function reformatUSPhone (USPhone)
{   return (reformat (USPhone, "(", 3, ") ", 3, "-", 4))
}



// checkUSPhone (TEXTFIELD theField [, BOOLEAN emptyOK==false])
//
// Check that string theField.value is a valid US Phone.
//
// For explanation of optional argument emptyOK,
// see comments of function isInteger.

function checkUSPhone (theField, emptyOK)
{   if (checkUSPhone.arguments.length == 1) emptyOK = defaultEmptyOK;
    if ((emptyOK == true) && (isEmpty(theField.value))) return true;
    else
    {  var normalizedPhone = stripCharsInBag(theField.value, phoneNumberDelimiters)
       if (!isUSPhoneNumber(normalizedPhone, false)) 
          return warnInvalid (theField, iUSPhone);
       else 
       {  // if you don't want to reformat as (123) 456-789, comment next line out
          theField.value = reformatUSPhone(normalizedPhone)
          return true;
       }
    }
}



// checkInternationalPhone (TEXTFIELD theField [, BOOLEAN emptyOK==false])
//
// Check that string theField.value is a valid International Phone.
//
// For explanation of optional argument emptyOK,
// see comments of function isInteger.

function checkInternationalPhone (theField, emptyOK)
{   if (checkInternationalPhone.arguments.length == 1) emptyOK = defaultEmptyOK;
    if ((emptyOK == true) && (isEmpty(theField.value))) return true;
    else
    {  if (!isInternationalPhoneNumber(theField.value, false)) 
          return warnInvalid (theField, iWorldPhone);
       else return true;
    }
}



// checkEmail (TEXTFIELD theField [, BOOLEAN emptyOK==false])
//
// Check that string theField.value is a valid Email.
//
// For explanation of optional argument emptyOK,
// see comments of function isInteger.

function checkEmail (theField, emptyOK)
{   if (checkEmail.arguments.length == 1) emptyOK = defaultEmptyOK;
    if ((emptyOK == true) && (isEmpty(theField.value))) return true;
    else if (!isEmail(theField.value, false)) 
       return warnInvalid (theField, iEmail);
    else return true;
}



// takes SSN, a string of 9 digits
// and reformats as 123-45-6789

function reformatSSN (SSN)
{   return (reformat (SSN, "", 3, "-", 2, "-", 4))
}


// Check that string theField.value is a valid SSN.
//
// For explanation of optional argument emptyOK,
// see comments of function isInteger.

function checkSSN (theField, emptyOK)
{   if (checkSSN.arguments.length == 1) emptyOK = defaultEmptyOK;
    if ((emptyOK == true) && (isEmpty(theField.value))) return true;
    else
    {  var normalizedSSN = stripCharsInBag(theField.value, SSNDelimiters)
       if (!isSSN(normalizedSSN, false)) 
          return warnInvalid (theField, iSSN);
       else 
       {  // if you don't want to reformats as 123-456-7890, comment next line out
          theField.value = reformatSSN(normalizedSSN)
          return true;
       }
    }
}




// Check that string theField.value is a valid Year.
//
// For explanation of optional argument emptyOK,
// see comments of function isInteger.

function checkYear (theField, emptyOK)
{   if (checkYear.arguments.length == 1) emptyOK = defaultEmptyOK;
    if ((emptyOK == true) && (isEmpty(theField.value))) return true;
    if (!isYear(theField.value, false)) 
       return warnInvalid (theField, iYear);
    else return true;
}


// Check that string theField.value is a valid Month.
//
// For explanation of optional argument emptyOK,
// see comments of function isInteger.

function checkMonth (theField, emptyOK)
{   if (checkMonth.arguments.length == 1) emptyOK = defaultEmptyOK;
    if ((emptyOK == true) && (isEmpty(theField.value))) return true;
    if (!isMonth(theField.value, false)) 
       return warnInvalid (theField, iMonth);
    else return true;
}


// Check that string theField.value is a valid Day.
//
// For explanation of optional argument emptyOK,
// see comments of function isInteger.

function checkDay (theField, emptyOK)
{   if (checkDay.arguments.length == 1) emptyOK = defaultEmptyOK;
    if ((emptyOK == true) && (isEmpty(theField.value))) return true;
    if (!isDay(theField.value, false)) 
       return warnInvalid (theField, iDay);
    else return true;
}



// checkDate (yearField, monthField, dayField, STRING labelString [, OKtoOmitDay==false])
//
// Check that yearField.value, monthField.value, and dayField.value 
// form a valid date.
//
// If they don't, labelString (the name of the date, like "Birth Date")
// is displayed to tell the user which date field is invalid.
//
// If it is OK for the day field to be empty, set optional argument
// OKtoOmitDay to true.  It defaults to false.

function checkDate (yearField, monthField, dayField, labelString, OKtoOmitDay)
{   // Next line is needed on NN3 to avoid "undefined is not a number" error
    // in equality comparison below.
    if (checkDate.arguments.length == 4) OKtoOmitDay = false;
    if (!isYear(yearField.value)) return warnInvalid (yearField, iYear);
    if (!isMonth(monthField.value)) return warnInvalid (monthField, iMonth);
    if ( (OKtoOmitDay == true) && isEmpty(dayField.value) ) return true;
    else if (!isDay(dayField.value)) 
       return warnInvalid (dayField, iDay);
    if (isDate (yearField.value, monthField.value, dayField.value))
       return true;
    alert (iDatePrefix + labelString + iDateSuffix)
    return false
}



// Get checked value from radio button.

function getRadioButtonValue (radio)
{   for (var i = 0; i < radio.length; i++)
    {   if (radio[i].checked) { break }
    }
    return radio[i].value
}




// Validate credit card info.

function checkCreditCard (radio, theField)
{   var cardType = getRadioButtonValue (radio)
    var normalizedCCN = stripCharsInBag(theField.value, creditCardDelimiters)
    if (!isCardMatch(cardType, normalizedCCN)) 
       return warnInvalid (theField, iCreditCardPrefix + cardType + iCreditCardSuffix);
    else 
    {  theField.value = normalizedCCN
       return true
    }
}



/*  ================================================================
    Credit card verification functions
    Originally included as Starter Application 1.0.0 in LivePayment.
    20 Feb 1997 modified by egk:
           changed naming convention to initial lowercase
                  (isMasterCard instead of IsMasterCard, etc.)
           changed isCC to isCreditCard
           retained functions named with older conventions from
                  LivePayment as stub functions for backward 
                  compatibility only
           added "AMERICANEXPRESS" as equivalent of "AMEX" 
                  for naming consistency 
    ================================================================ */


/*  ================================================================
    FUNCTION:  isCreditCard(st)
 
    INPUT:     st - a string representing a credit card number

    RETURNS:  true, if the credit card number passes the Luhn Mod-10
                    test.
              false, otherwise
    ================================================================ */

function isCreditCard(st) {
  // Encoding only works on cards with less than 19 digits
  if (st.length > 19)
    return (false);

  sum = 0; mul = 1; l = st.length;
  for (i = 0; i < l; i++) {
    digit = st.substring(l-i-1,l-i);
    tproduct = parseInt(digit ,10)*mul;
    if (tproduct >= 10)
      sum += (tproduct % 10) + 1;
    else
      sum += tproduct;
    if (mul == 1)
      mul++;
    else
      mul--;
  }
// Uncomment the following line to help create credit card numbers
// 1. Create a dummy number with a 0 as the last digit
// 2. Examine the sum written out
// 3. Replace the last digit with the difference between the sum and
//    the next multiple of 10.

//  document.writeln("<BR>Sum      = ",sum,"<BR>");
//  alert("Sum      = " + sum);

  if ((sum % 10) == 0)
    return (true);
  else
    return (false);

} // END FUNCTION isCreditCard()



/*  ================================================================
    FUNCTION:  isVisa()
 
    INPUT:     cc - a string representing a credit card number

    RETURNS:  true, if the credit card number is a valid VISA number.
                    
              false, otherwise

    Sample number: 4111 1111 1111 1111 (16 digits)
    ================================================================ */

function isVisa(cc)
{
  if (((cc.length == 16) || (cc.length == 13)) &&
      (cc.substring(0,1) == 4))
    return isCreditCard(cc);
  return false;
}  // END FUNCTION isVisa()




/*  ================================================================
    FUNCTION:  isMasterCard()
 
    INPUT:     cc - a string representing a credit card number

    RETURNS:  true, if the credit card number is a valid MasterCard
                    number.
                    
              false, otherwise

    Sample number: 5500 0000 0000 0004 (16 digits)
    ================================================================ */

function isMasterCard(cc)
{
  firstdig = cc.substring(0,1);
  seconddig = cc.substring(1,2);
  if ((cc.length == 16) && (firstdig == 5) &&
      ((seconddig >= 1) && (seconddig <= 5)))
    return isCreditCard(cc);
  return false;

} // END FUNCTION isMasterCard()





/*  ================================================================
    FUNCTION:  isAmericanExpress()
 
    INPUT:     cc - a string representing a credit card number

    RETURNS:  true, if the credit card number is a valid American
                    Express number.
                    
              false, otherwise

    Sample number: 340000000000009 (15 digits)
    ================================================================ */

function isAmericanExpress(cc)
{
  firstdig = cc.substring(0,1);
  seconddig = cc.substring(1,2);
  if ((cc.length == 15) && (firstdig == 3) &&
      ((seconddig == 4) || (seconddig == 7)))
    return isCreditCard(cc);
  return false;

} // END FUNCTION isAmericanExpress()




/*  ================================================================
    FUNCTION:  isDinersClub()
 
    INPUT:     cc - a string representing a credit card number

    RETURNS:  true, if the credit card number is a valid Diner's
                    Club number.
                    
              false, otherwise

    Sample number: 30000000000004 (14 digits)
    ================================================================ */

function isDinersClub(cc)
{
  firstdig = cc.substring(0,1);
  seconddig = cc.substring(1,2);
  if ((cc.length == 14) && (firstdig == 3) &&
      ((seconddig == 0) || (seconddig == 6) || (seconddig == 8)))
    return isCreditCard(cc);
  return false;
}



/*  ================================================================
    FUNCTION:  isCarteBlanche()
 
    INPUT:     cc - a string representing a credit card number

    RETURNS:  true, if the credit card number is a valid Carte
                    Blanche number.
                    
              false, otherwise
    ================================================================ */

function isCarteBlanche(cc)
{
  return isDinersClub(cc);
}




/*  ================================================================
    FUNCTION:  isDiscover()
 
    INPUT:     cc - a string representing a credit card number

    RETURNS:  true, if the credit card number is a valid Discover
                    card number.
                    
              false, otherwise

    Sample number: 6011000000000004 (16 digits)
    ================================================================ */

function isDiscover(cc)
{
  first4digs = cc.substring(0,4);
  if ((cc.length == 16) && (first4digs == "6011"))
    return isCreditCard(cc);
  return false;

} // END FUNCTION isDiscover()





/*  ================================================================
    FUNCTION:  isEnRoute()
 
    INPUT:     cc - a string representing a credit card number

    RETURNS:  true, if the credit card number is a valid enRoute
                    card number.
                    
              false, otherwise

    Sample number: 201400000000009 (15 digits)
    ================================================================ */

function isEnRoute(cc)
{
  first4digs = cc.substring(0,4);
  if ((cc.length == 15) &&
      ((first4digs == "2014") ||
       (first4digs == "2149")))
    return isCreditCard(cc);
  return false;
}



/*  ================================================================
    FUNCTION:  isJCB()
 
    INPUT:     cc - a string representing a credit card number

    RETURNS:  true, if the credit card number is a valid JCB
                    card number.
                    
              false, otherwise
    ================================================================ */

function isJCB(cc)
{
  first4digs = cc.substring(0,4);
  if ((cc.length == 16) &&
      ((first4digs == "3088") ||
       (first4digs == "3096") ||
       (first4digs == "3112") ||
       (first4digs == "3158") ||
       (first4digs == "3337") ||
       (first4digs == "3528")))
    return isCreditCard(cc);
  return false;

} // END FUNCTION isJCB()



/*  ================================================================
    FUNCTION:  isAnyCard()
 
    INPUT:     cc - a string representing a credit card number

    RETURNS:  true, if the credit card number is any valid credit
                    card number for any of the accepted card types.
                    
              false, otherwise
    ================================================================ */

function isAnyCard(cc)
{
  if (!isCreditCard(cc))
    return false;
  if (!isMasterCard(cc) && !isVisa(cc) && !isAmericanExpress(cc) && !isDinersClub(cc) &&
      !isDiscover(cc) && !isEnRoute(cc) && !isJCB(cc)) {
    return false;
  }
  return true;

} // END FUNCTION isAnyCard()



/*  ================================================================
    FUNCTION:  isCardMatch()
 
    INPUT:    cardType - a string representing the credit card type
              cardNumber - a string representing a credit card number

    RETURNS:  true, if the credit card number is valid for the particular
              credit card type given in "cardType".
                    
              false, otherwise
    ================================================================ */

function isCardMatch (cardType, cardNumber)
{

        cardType = cardType.toUpperCase();
        var doesMatch = true;

        if ((cardType == "VISA") && (!isVisa(cardNumber)))
                doesMatch = false;
        if ((cardType == "MASTERCARD") && (!isMasterCard(cardNumber)))
                doesMatch = false;
        if ( ( (cardType == "AMERICANEXPRESS") || (cardType == "AMEX") )
                && (!isAmericanExpress(cardNumber))) doesMatch = false;
        if ((cardType == "DISCOVER") && (!isDiscover(cardNumber)))
                doesMatch = false;
        if ((cardType == "JCB") && (!isJCB(cardNumber)))
                doesMatch = false;
        if ((cardType == "DINERS") && (!isDinersClub(cardNumber)))
                doesMatch = false;
        if ((cardType == "CARTEBLANCHE") && (!isCarteBlanche(cardNumber)))
                doesMatch = false;
        if ((cardType == "ENROUTE") && (!isEnRoute(cardNumber)))
                doesMatch = false;
        return doesMatch;

}  // END FUNCTION CardMatch()




/*  ================================================================
    The below stub functions are retained for backward compatibility
    with the original LivePayment code so that it should be possible
    in principle to swap in this new module as a replacement for the  
    older module without breaking existing code.  (There are no
    guarantees, of course, but it should work.)

    When writing new code, do not use these stub functions; use the
    functions defined above.
    ================================================================ */

function IsCC (st) {
    return isCreditCard(st);
}

function IsVisa (cc)  {
  return isVisa(cc);
}

function IsVISA (cc)  {
  return isVisa(cc);
}

function IsMasterCard (cc)  {
  return isMasterCard(cc);
}

function IsMastercard (cc)  {
  return isMasterCard(cc);
}

function IsMC (cc)  {
  return isMasterCard(cc);
}

function IsAmericanExpress (cc)  {
  return isAmericanExpress(cc);
}

function IsAmEx (cc)  {
  return isAmericanExpress(cc);
}

function IsDinersClub (cc)  {
  return isDinersClub(cc);
}

function IsDC (cc)  {
  return isDinersClub(cc);
}

function IsDiners (cc)  {
  return isDinersClub(cc);
}

function IsCarteBlanche (cc)  {
  return isCarteBlanche(cc);
}

function IsCB (cc)  {
  return isCarteBlanche(cc);
}

function IsDiscover (cc)  {
  return isDiscover(cc);
}

function IsEnRoute (cc)  {
  return isEnRoute(cc);
}

function IsenRoute (cc)  {
  return isEnRoute(cc);
}

function IsJCB (cc)  {
  return isJCB(cc);
}

function IsAnyCard(cc)  {
  return isAnyCard(cc);
}

function IsCardMatch (cardType, cardNumber)  {
  return isCardMatch (cardType, cardNumber);
}

/////////////////////////////////////////////////////////////////////
// validate numeric value
// parameters:
//		input		input string to validate
//		fmt			format	UNM		unsigned numeric, no punctuation
//		prec		total number of digits allowed
//		scale		maximum number of fractional digits allowed
//		req			is this required or is blank ok
//
function isValidNumber(input, fmt, prec, scale, req)
{
	var			dpos = 0
	var			sLeft = ""
	var			sRight = ""
	var			i = 0
	var			comma = ","
	var			decpnt = "."

	// first validate input parameters
	if (fmt != "UNM" && fmt != "UNS" && fmt != "PNM" && fmt != "PNS" &&
		fmt != "ENM" && fmt != "ENS")
	{
		alert("Invalid format: " + fmt)
		return false;
	}
	if (!isPositiveInteger(new String(prec)))
	{
		alert("Invalid precision: " + prec + 
			", must be a number greater than zero")
		return false;
	}
	if (!isPositiveInteger(new String(scale)))
	{
		alert("Invalid scale: " + scale + 
			", must be a number greater than or equal to zero")
		return false;
	}
	if (!isYesOrNo(req))
	{
		alert("Invalid required flag: " + req)
		return false;
	}

	// set comma and decimal point characters
	if (fmt.charAt(0) == "E")
	{
		decpnt = ","
		comma = "."
	}

	// create string object for input
	sNumber = new String(input.value)

	// locate decimal point
	dpos = sNumber.indexOf(decpnt)

	// get digits before and after decimal point
	if (dpos > -1)
	{
		sLeft = sNumber.substring(0, dpos)
		sRight = sNumber.substring(dpos + 1, sNumber.length)

	}
	else
	{
		sLeft = sNumber
		sRight = ""
	}

	// check punctuation
	j = 0
	for (i = sLeft.length - 1; i >= 1; i--)
	{
		j++
		if (sLeft.charAt(i) == comma)
		{
			if (fmt.charAt(0) != "P" && fmt.charAt(0) != "E")
			{
				alert("Punctuation not allowed")
				return false
			}
			else if (j != 4)
			{
				alert("Invalid punctuation: " + sNumber)
				return false
			}
			j = 0
		}
		if (j == 4)
			j = 1
	}

	// remove punctuation
	sNumber = stripCharsInBag(sNumber, comma)

	// check the value
	if (fmt.charAt(fmt.length - 1) == "M")
	{
		if (!isFloat(sNumber, true, decpnt))
		{
			alert("Invalid number: " + sNumber)
			return false;
		}
	}
	else
	{
		if (!isSignedFloat(sNumber, true, decpnt))
		{
			alert("Invalid number: " + sNumber)
			return false;
		}
	}

	if (sNumber.length == 0)
	{
		if (req.toUpperCase() == "N")
			return true;
		else
		{
			alert("Enter a number, entry is required")
			return false;
		}
	}

	// validate length of fraction
	if (sRight.length > parseInt(scale))
	{
		alert("Fraction too long, " + scale + " digit(s) allowed")
		return false;
	}

	// validate length
	i = sNumber.length
	if (dpos > -1)
		i -= 1
	if (i > prec)
	{
		alert("Number is too large, format is " + prec + ", " + scale)
		return false;
	}
	return true;
}
/////////////////////////////////////////////////////////////////////
// validate time
// parameters:
//		input		input string to validate
//		req			is this required or is blank ok
//
function isValidTime(input, req)
{
	var			idx = 0
	var			hour = 0
	var			min = 0
	var			sec = 0
	var			pos = 0

	// create string object for input
	sVal = new String(input.value)
	stripWhitespace(sVal)

	// check for blank
	if (isEmpty(sVal))
	{
		if (req.toUpperCase() == "N")
			return true
		else
		{
			alert("Enter a time, entry is required")
			return false
		}
	}

	// locate colon, validate hour
	idx = sVal.indexOf(":")
	if (idx > 2 || idx < 1)
	{
		alert("Invalid time: " + input.value)
		return false
	}

	hour = sVal.substring(0, idx)
	if (!isIntegerInRange(hour, 0, 23))
	{
		alert("Invalid hours: " + hour)
		return false
	}

	// validate minutes
	pos = idx + 1
	sVal = sVal.substring(pos)
	if (sVal.length == 0)
	{
		alert("Minutes are required")
		return false
	}

	idx = sVal.indexOf(":")
	if (idx == -1)
		min = sVal
	else
		min = sVal.substring(0, idx)

	if (!isIntegerInRange(min, 0, 59))	 
	{
		alert("Invalid minutes: " + min)
		return false
	}

	// validate seconds
	if (idx > -1)
	{
		pos = idx + 1
		sec = sVal.substring(pos)
		if (!isIntegerInRange(sec, 0, 59, true))
		{
			alert("Invalid seconds: " + sec)
			return false
		}
	}
	return true
}
//----------------------------------------------------------------
// Entered by Karen Zelevinsky and Helen Taycher of PegaSystems on February 12, 1999


// Display a layer/div
function showLayer(layerName) {
	if (navigator.appName == "Netscape") {
		document.layers[layerName].visibility="visible";		
	} else {
		document.all[layerName].style.visibility="visible" }
}

// Hide a layer/div
function hideLayer(layerName) {
	if (navigator.appName == "Netscape") {
		document.layers[layerName].visibility="hidden";
	} else {
		document.all[layerName].style.visibility="hidden" }	
} 

// If conditional statement (thisCase) is true, call showLayer function and return true.
// If conditional statement (thisCase) is false, call hideLayer and return false.  
function flagOnSubmit(thisCase, layer) {
	if (thisCase) {
		showLayer(layer);
		return true;	
	} else {
		hideLayer(layer);
		return false;
	}
}

// Call showLayer if the text object is empty and hideLayer if it is not empty.		
function flagEmptyField(textObj, layer) {
	if ((textObj.value == null) || (textObj.value.length == 0)) {
		showLayer(layer);
	} else {	
		hideLayer(layer);
	} 
}

// Call showLayer if Social Security Number is not valid and hideLayer if it is valid.
function flagNonZipCode(textObj, layer){
	if (isValidZipCode(textObj)) {
		hideLayer(layer);
	} else {
		showLayer(layer);
	}
}

// Call showLayer if state code is not valid and hideLayer if it is valid.
function flagNonState(textObj, layer){
	var result = isStateCode(textObj.value);
	if (result == false) {showLayer(layer)}
	else {hideLayer(layer)}
}

// Call showLayer if Year  is not valid and hideLayer if it is valid.
function flagNonYear(textObj, layer){
	if (isYear(textObj.value)) {
		hideLayer(layer);
	} else {
		showLayer(layer);
	}
}

// Call showLayer if credit card number  is not valid and hideLayer if it is valid.
function flagNonCreditCard(textObj, layer){
	if (isCreditCard(textObj.value)) {
		hideLayer(layer);
	} else {
		showLayer(layer);
	}
}

// Call showLayer if phone number  is not valid and hideLayer if it is valid.
function flagNonUSPhoneNumber(textObj, layer){
	if (isValidPegaPhone(textObj.value)) {
		hideLayer(layer);
	} else {
		showLayer(layer);
	}
}

// Call showLayer if phone number  is not valid and hideLayer if it is valid.
function flagNonUS_SSN(textObj, layer){
	if (isSSN(textObj.value)) {
		hideLayer(layer);
	} else {
		showLayer(layer);
	}
}

// Call showLayer if international phone number  is not valid and hideLayer if it is valid.
function flagNonIntrPhoneNumber(textObj, layer){
	if (isInternationalPhoneNumber(textObj.value)) {
		hideLayer(layer);
	} else {
		showLayer(layer);
	}
}

// Call showLayer if Day  is not valid and hideLayer if it is valid.
function flagNonDay(textObj, layer){
	if (isPegaDay(textObj.value)) {
		hideLayer(layer);
	} else {
		showLayer(layer);
	}
}

// Call showLayer if Month   is not valid and hideLayer if it is valid.
function flagNonMonth(textObj, layer){
	if (isPegaMonth(textObj.value)) {
		hideLayer(layer);
	} else {
		showLayer(layer);
	}
}

// Call showLayer if date   is not valid and hideLayer if it is valid.
function flagNonDate(year,month,day, layer){
	if (isDate(year,month,day)) {
		hideLayer(layer);
	} else {
		showLayer(layer);
	}
}

// Call showLayer if e-mail  is not valid and hideLayer if it is valid.
function flagNonEMail(textObj, layer){
	if (isEmail(textObj.value)) {
		hideLayer(layer);
	} else {
		showLayer(layer);
	}
}

// Call showLayer if visa credit card number  is not valid and hideLayer if it is valid.
function flagNonVisaCC(textObj, layer){
	if (isVisa(textObj.value)) {
		hideLayer(layer);
	} else {
		showLayer(layer);
	}
}

// Call showLayer if Maaster credit card number  is not valid and hideLayer if it is valid.
function flagNonMasterCC(textObj, layer){
	if (isMasterCard(textObj.value)) {
		hideLayer(layer);
	} else {
		showLayer(layer);
	}
}

// Call showLayer if american credit card number  is not valid and hideLayer if it is valid.
function flagNonAmericanCC(textObj, layer){
	if (isAmericanExpress(textObj.value)) {
		hideLayer(layer);
	} else {
		showLayer(layer);
	}
}

// Call showLayer if discovery credit card number  is not valid and hideLayer if it is valid.
function flagNonDiscoverCC(textObj, layer){
	if (isDiscover(textObj.value)) {
		hideLayer(layer);
	} else {
		showLayer(layer);
	}
}

// If conditional statement (thisCase) is false, call showLayer function and return true.
// If conditional statement (thisCase) is true, call hideLayer and return false.  
function flagErrorsOnSubmit(thisCase, layer) {
	if (thisCase) {
		hideLayer(layer);
		return true;
		
	} else {
		showLayer(layer);
		return false;	
	}
}

// Returns false if text object is empty and true if text object is not empty.
function isCompletedField(s) {	
	if ((s == null) || (s.length == 0)) {
		return false;
	} else {
		return true;
	}
}

function isPegaMonth(s) {
  	if (s == null || s.length == 0) {
		return false;
	} else {
		var sNum =parseInt(s, 10);
		if (sNum >=1 && sNum <=12) {
			return true;
		} else {
			return false;
		}
	}
}

function isPegaDay(s) {
  	if (s == null || s.length == 0) {
		return false;
	} else {
		var sNum =parseInt(s, 10);
		if (sNum >=1 && sNum <=31) {
			return true;
		} else {
			return false;
		}
	}
}

function isValidPegaPhone(s){
	var thisPhone = stripCharsInBag(s, phoneNumberDelimiters);
	
	var result0 = thisPhone.charAt(3);
	var result1 = isPositiveInteger(thisPhone);
	var result2 = (thisPhone.length == digitsInUSPhoneNumber);
	
		if (result1 == false || result2 == false || result1 == 0 || result0 == "0" ) {return false;}
		else {return true;}
}

// Convert text to all upper case.
function makeUpperCase(textObj) {
	var theText = textObj.value;		
	var upperCase = theText.toUpperCase();
	textObj.value = upperCase;
}	

// The functions below this are here for historical reasons and are not optimal

// Returns true if text object is empty and false if text object is not empty.
function isEmptyField(textObj) {	
	if ((textObj.value == null) || (textObj.value.length == 0)) {
		return true;
	} else {
		return false;
	}
}

function emptyField(textObj, layer) {
	if ((textObj.value == null) || (textObj.value.length == 0)) {
		showLayer(layer);
	} else {	
		hideLayer(layer);
	} 

}

function do_number(textObj, layer) {
 	if (isPositiveInteger(textObj.value)) {
		hideLayer(layer)
	} else {
		showLayer(layer)
	}
}

function do_state(textObj, layer) {
	if (isStateCode(textObj.value)) {
		hideLayer(layer)
	} else {
		showLayer(layer)

	}
}

function isValidZipCode(textObj){
	var thisZipCode = stripCharsInBag(textObj.value, ZIPCodeDelimiters);
	var result1 = isPositiveInteger(thisZipCode);
	var result2 = (thisZipCode.length == digitsInZIPCode1 || thisZipCode.length == digitsInZIPCode2);

		if (result1 == false || result2 == false) {return false;}
		else {return true;}
}


// FUNCTIONS TO ENABLE AND DISABLE FIELDS
//
// enableField(fieldName)		Enables field and sets background to white
// enableAllFields(arrayOfFieldNames)	Enables all fields that match the names in the array
// disableField(fieldName [,divName])	Disables field, sets bgcolor gray, and hides error
// disableOtherFields(enabledFieldName,	Calls disableField on all fields except the one  
//   arrayOfFieldNames,arrayOfDivNames)	  that matches the enabledFieldName


// BASIC VALIDATION FUNCTIONS
//
// isValidPostalCode(s)			True if s is a valid zip code (strips delimiters)
// isAlphabeticWithSpaces(s[,emptyOK])	True if s is English letters or a space
// isAlphanumericWithSpaces(s[,emptyOK])True if s is English letters, numbers, or a space


// OTHER
//
// createFourDigitYear(twoDigitYear)	Creates a 4 digit year from a 2 digit year
// formatAOLPhoneNumber(s,textObj)	Inserts formatted phone number as value of textObj
// limitTextArea(myArea,myLength) 	limits textarea size (as user types)
// checkTextAreaLength(myArea,myLength)	limits textarea size (for onblur event)


// VALIDATION FUNCTIONS THAT ALSO REFORMAT DATA IN ORIGINAL FIELD
//
// isAOLDate(textObj)			True if textObj.value is date-Reformat to MM-DD-YYYY
// isAOLCCDate(textObj)			True if textObj.value is a credit card date.   
//                                        Reformats to MM-YYYY or MM-DD-YYYY
// isAOLTime(textObj)			True if textObj.value is a time-Reformats to HH:MM
// isAOLAmount(textObj 			True if textObj.value is an amount-Reformats to D.CC
//   [,centsRequired,maxLength])
// isAOLPhoneNumber(textObj		True if textObj.value is a US phone number-Reformats
//   ,hiddenInputName)			  to (NNN) NNN-NNNN and hidden input to NNNNNNNNNN  
//                                                 

// VALIDATION FUNCTIONS THAT FLAG ERRORS
// 
// flagNonText(textObj,layer		Shows error if isAlphabeticWithSpaces(textObj.value) 
//   [,requiredSize])	 		  returns false
// flagNonAOLDate(textObj,layer)	Shows error if isAOLDate(textObj) returns false
// flagNonAOLCCDate(textObj,layer)	Shows error if isAOLCCDate(textObj) returns false
// flagNonAOLAmount(textObj,layer)	Shows error if isAOLAmount(textObj) returns false
// flagNonInteger(textObj,layer)	Shows error if isInteger(textObj.value) is false
// flagNonAOLPhoneNumber(textObj,	Shows error if isAOLPhoneNumber(textObj,
//    layer,hiddenInputName)		  hiddenInputName) returns false


// VALIDATION FUNCTIONS THAT DETERMINE WHETHER FORM SHOULD BE SUBMITTED
// 
// validateInput(thisForm)		Returns true if all inputs in thisForm with 
// 					  validation functions return true 
// validateListSelection(listID)	Returns true if a row from listID PegaList is // 					  selected


// GLOBAL VARIABLE DECLARATIONS

var cancelButton = false; // Determines whether validation fields should be checked.
			  //    When value is true, no checks are required. This will
                          //    happen when a submit button is a cancel or back button. 





// Disables the field with name fieldName, erases any value, and sets the background color 
// of the input to silver. If a second parameter to the function exists, calls hideLayer 
// which hides the associated error flag.

function disableField(fieldName,divName) 
{
   document.all(fieldName).disabled = true;
   document.all(fieldName).value = ""; // Blanks out the value of the field
   document.all(fieldName).style.backgroundColor = "silver";
   if (disableField.arguments.length == 2) // If a divName was passed to the function
   {
      if (divName != "") // If divName is not blank
      {
         hideLayer(divName);
      }
   }
}


// Enables field with name fieldName and sets the background color of the input to white.

function enableField(fieldName,divName) 
{
   document.all(fieldName).disabled = false;
   document.all(fieldName).style.backgroundColor = "white";
   if (enableField.arguments.length == 2)
   {
      if (divName != "") // If divName is not blank
      {
         hideLayer(divName);
      }
   }
}


// For each member of the array of field names: erases any value in the field and calls
// enableField on that field.

function enableAllFields(arrayOfFieldNames, arrayOfDivNames)
{
   for(var i=0; i<arrayOfFieldNames.length; i++) // For every member of the array
   {
      if (arrayOfFieldNames[i] != "") // If a value exists
      {
         document.all(arrayOfFieldNames[i]).value = ""; // Blanks out the value of the field
         if (enableAllFields.arguments.length == 2 && arrayOfDivNames[i] != "") 
         // If there is an associated div with a non-blank value
         {
   	    enableField(arrayOfFieldNames[i], arrayOfDivNames[i]);
         } else
         {
            enableField(arrayOfFieldNames[i]);
         }
      }
   }
}


// For each member of the array of field names: if it does not match enabledFieldName,
// calls disableField on that field. 

function disableOtherFields(enabledFieldName, arrayOfFieldNames, arrayOfDivNames)
{
   for(var i=0; i<arrayOfFieldNames.length; i++) // For every member of the array
   {
      if (arrayOfFieldNames[i] != "") // If a value exists
      {  
         if (arrayOfFieldNames[i] != enabledFieldName)
         // If the value doesn't match the string enabledFieldName
         {
            if (arrayOfDivNames[i] == "") // If the associated div name is blank
            {
               disableField(arrayOfFieldNames[i]);
            } else
            {
               disableField(arrayOfFieldNames[i],arrayOfDivNames[i]); 
            }
         }
      }
   }
}


// Returns true if s is a valid zip code. 

function isValidPostalCode(s)
{
   var ZIPCode = stripCharsInBag(s, ZIPCodeDelimiters); // Strips delimiters
   if (isZIPCode(s)) 
   {
      return true;
   } else 
   {
      return false;
   }
}

// Returns true if s contains only English letters and spaces

function isAlphabeticWithSpaces (s, emptyOkay)
{   
   s = stripWhitespace(s); // Strip spaces from the string
   if (isAlphabeticWithSpaces.arguments.length == 1) // If no emptyOkay value was passed
   {
      emptyOkay = defaultEmptyOK;
   } 
   return (isAlphabetic(s,emptyOkay));
}


// Returns true if s contains only English letters, numbers, and spaces

function isAlphanumericWithSpaces (s, emptyOkay)
{   
   s = stripWhitespace(s); // Strips spaces from the string
   if (isAlphanumericWithSpaces.arguments.length == 1) 
   {
      emptyOkay = defaultEmptyOK;
   }
   return (isAlphanumeric(s,emptyOkay));
}


// Returns true if textObj.value is a valid date. The original data can take many formats:
// MMDDYY, MM-DD-YY, M/D/YYYY are all valid. The data is reformatted as MM-DD-YYYY and
// is inserted back into the text object.

function isAOLDate(textObj)
{
   var date = textObj.value;
   var delimiter, day, month, year, firstDelimiter, secondDelimiter;
   var hasDelimiter = false;

   if (date.indexOf("-") > 0) // If the date contains delimiter "-"
   {
      delimiter = "-";
      hasDelimiter = true;
   } else if (date.indexOf("/") > 0) // If the date contains delimiter "/"
   {
      delimiter = "/";
      hasDelimiter = true;
   }
   if (hasDelimiter)
   {
      firstDelimiter = date.indexOf(delimiter); // First instance of the delimiter
      secondDelimiter = date.indexOf(delimiter,firstDelimiter+1); // Second instance
      if (firstDelimiter > 0 && secondDelimiter > 0) // If both delimitors exist
      {
         month = date.substring(0,firstDelimiter);
         day = date.substring(firstDelimiter+1,secondDelimiter);
         year = date.substring(secondDelimiter+1,date.length);
      } else
      {
         return false;
      }
   } else // If there are no delimiters
   {
      if (date.length == 6 || date.length == 8) 
      {
         month = date.substring(0,2);
         day = date.substring(2,4);
         year = date.substring(4,date.length);
      } else
      {
         return false;
      }
   }
   if (isDate(year,month,day))
   {
      if (month.length == 1) // If single digit month (e.g., 2 instead of 02)
      {
         month = "0" + month;
      }
      if (day.length == 1) // If single digit day (e.g., 5 instead of 05)
      {
         day = "0" + day;
      }
      if (year.length==2) // If two digit year (e.g., 03 instead of 2003)
      {
         year = createFourDigitYear(year);
      }

      textObj.value = (month+"-"+day+"-"+year); // Set value of input to formatted date
      return true;
   } else
   {
      return false;   
   }
}


// Returns true if textObj.value is a valid credit card date. The original data can take 
// many formats: MMDDYYYY, M-YY, MM/YYYY are all valid. The data is reformatted as 
// either MM-DD-YYYY or MM-YYYY and is inserted back into the text object.
//
// NOTE: There is possible confusion between MMYYYY and MMDDYY because they both contain
// 6 digits and have no delimiters. Therefore these are not valid input.

function isAOLCCDate(formattedTextObj,nonFormattedInputName)
{
   var date = formattedTextObj.value;
   var delimiter, month, year, firstDelimiter;
   var hasDelimiter = false;

   if (date.indexOf("-") > 0) // If the date contains delimiter "-"
   {
      delimiter = "-";
      hasDelimiter = true;
   } else if (date.indexOf("/") > 0) // If the date contains delimiter "/"
   {
      delimiter = "/";
      hasDelimiter = true;
   }
   if (hasDelimiter)
   {
      firstDelimiter = date.indexOf(delimiter); // First instance of the delimiter
      if (firstDelimiter > 0) // If delimiter exists
      {
         month = date.substring(0,firstDelimiter);
         year = date.substring(firstDelimiter+1,date.length);
      } else
      {
         return false;
      }
   } else // If there are no delimiters
   {
      if (date.length == 4) // If date is 4 digits (e.g., MMYY)
      {
         month = date.substring(0,2);
         year = date.substring(2,4);
      } else if (date.length == 6) // If date is 6 digits (e.g, MMYYYY)
      {
         month = date.substring(0,2);
         year = date.substring(2,6);
      } else
      {
         return false;
      }
   }
   if (isYear(year) && isMonth(month))
   {
      if (month.length == 1) // If single digit month (e.g., 2 instead of 02)
      {
         month = "0" + month;
      }
      if (year.length == 2) // If year is two digits (e.g., 03 instead of 2003)
      {
         year = createFourDigitYear(year);
      }
      formattedTextObj.value = month + "-" + year;
      document.all(nonFormattedInputName).value = month + (year.substring(2,4));
      return true;
   } else
   {
      return false;
   }
}


// Returns a four digit year given a two digit year. This is done by calculating a 100 year
// range around the current year. For example, if the current year is 2004, the 100 year
// range would be 1955 - 2054. If the two digit year to be converted is between 55 and 99,
// it is part of the previous century ("19"). If the two digit year is between 00 and 54, it 
// is part of the current century ("20"). As a second example, if the current year is 2065, 
// the 100 year range is 2016 - 2115. If the two digit year is between 16 and 99, it is 
// part of the current century ("20"). If it is between 00 and 15, it is part of the future 
// century ("21").

function createFourDigitYear(twoDigitYear)
{
   var thisCentury, lowerCentury, higherCentury = "";
   var today = new Date();
   var thisYear = (today.getFullYear()).toString();
   thisCentury = parseInt(thisYear.substring(0,2));
   if (thisYear.substring(2,4) > 50) // If the last two digits of the current year > 50
   {
      higherCentury = thisCentury + 1; // The 100 years are in this century and a higher one
   } else // If the last two digits of the current year <= 50
   {
      lowerCentury = thisCentury - 1; // The 100 years are in a lower century and this one
   }
   var startRange = parseInt(thisYear) - 49; // The first year of 100 years is this year-49
   var shortStartRange = (startRange.toString()).substring(2,4); // Last two digits of first year
   if (higherCentury != "") // If the range is in this century and a higher one
   {
      if (twoDigitYear < parseInt(shortStartRange)) // If year to be converted < first year
      {
         return (higherCentury.toString() + twoDigitYear); // year is in higher century
      } else // if year to be converted >= first year 
      {
         return (thisCentury.toString() + twoDigitYear); // year is in this century
      }
   } else // If the range is in a lower century and this one
   {
      if (twoDigitYear < parseInt(shortStartRange)) // If year to be converted < first year
      {
         return (thisCentury.toString() + twoDigitYear); // year is in this century

      } else // If year to be converted >= first year
      {
         return (lowerCentury.toString() + twoDigitYear); // year is in lower century
      }
   } 
}


// Returns true if textObj.value is a time (whole numbers are converted to hours). Also 
// reformats the time to HH:MM. A number of minutes between 60 and 99 is converted to 
// standard hour/minutes format.

function isAOLTime(textObj)
{
   var hours,minutes;
   var time = stripWhitespace(textObj.value);
   if (time.length==0)
   {
      return false;
   }
   var index = time.indexOf(":");
   if (index > 0)
   {
      hours = time.substring(0,index);
      minutes = time.substring(index+1,time.length);
   } 
   if (index == 0 || hours == "00" || hours == "0")
   {
      minutes = time.substring(index+1,time.length);
      if (minutes.length == 2)
      {
         if (minutes < 60)
         {
            hours = "00";
         } else
         {
            hours = "01";
            minutes = parseInt(minutes - 60).toString();
         }
      } else
      {
         return false;
      }     
   } else
   { 
      if (time <= 99)
      {
         hours = time;
         minutes = "00";
      } else if (time >= 100 && time <= 9999)
      {
         hours = parseInt(time/100).toString();
         minutes = parseInt(time%100).toString();
      } else if (time > 9999)
      {
         return false;
      }
   }
   if (hours.length==1)
   {
      hours = "0" + hours;
   }
   if (minutes.length==1)
   {
      minutes = "0" + minutes;
   }
   if (hours.length == 2 && isNonnegativeInteger(hours) 
       && minutes.length == 2 && isNonnegativeInteger(minutes) && minutes < 60)
   {
      textObj.value = hours + ":" + minutes;
      return true;
   } else
   {
      return false;
   }
}


// Returns true if textObj.value is an amount. Two optional parameters are centsRequired
// and maxLength. Reformats the amount to D.CC. A number with no decimal is converted to // dollars.

function isAOLAmount(formattedTextObj,nonFormattedInputName,centsRequired,maxLength)
{
   var amount = stripCharsInBag(formattedTextObj.value,"$, "); // Strip extra characters
   if (isCompletedField(amount) == false)
   {
      return false;
   }

   var hasCents = false;
   var decimal = amount.indexOf("."); // Find location of decimal
   if (decimal>=0) // If there is a decimal
   {
      var centAmount = amount.substring(decimal+1,amount.length); // Get num after decimal
      if (centAmount.length == 1) // If there is only 1 digit
      {
         centAmount = centAmount + "0"; // Add a 0 (e.g., 3 becomes 30)
      }
      if (centAmount.length == 2 && isNonnegativeInteger(centAmount))
      {
         hasCents = true;
         if (decimal == 0) // If there is nothing before the decimal (e.g., .50)
         {
            amount = "0"; // Dollar amount is 0 
         } else
         {
            amount = amount.substring(0,decimal); // Dollar amount is num before decimal
         }
      } else
      {
         return false;
      }
   }
   if (centsRequired==true && hasCents==false)
   {
      return false;
   }
   if (isNonnegativeInteger(maxLength))
   {
      if(amount.length+3 > maxLength) // If dollar amount + decimal and cents > maxLength
      {
         return false;
      }
   }
   if (hasCents==false) // If no cents were entered
   {
         centAmount = "00"; 
   }
   if (isNonnegativeInteger(amount)) // If amount is a number
   {

      var nonFormattedAmount = amount + centAmount;
      var formattedAmount = amount;
      var numDigitsInAmount = amount.length;
      if (numDigitsInAmount>3) // If commas are necessary
      {
         var numCommas = Math.floor(numDigitsInAmount/3);
         var lengthOfFirstPart = numDigitsInAmount%3; // Num digits before first comma
         if (lengthOfFirstPart == 0) // If number of digits divides evenly into 3
         {
            lengthOfFirstPart = 3;
            numCommas--;
         }
         var formattedAmount = amount.substring(0,lengthOfFirstPart); // Get first digits
         amount = amount.substring(lengthOfFirstPart,amount.length); // Rest of digits
         for (var i=0; i<numCommas; i++)
         {
            formattedAmount += "," + amount.substring(0,3); // Add comma and get next 3 nums
            amount = amount.substring(3,amount.length); // Rest of the digits
         }
      } 
      formattedAmount += "." + centAmount; // Add decimal and cent amount
      formattedTextObj.value = formattedAmount; // Set value of the text object
      document.all(nonFormattedInputName).value = nonFormattedAmount;
      return true;
   } else
   {
      return false;
   }
}


// Hide layer that contains the error image if textObj.value is alphabetic with spaces.
// Otherwise, show layer. 

function flagNonText(textObj, layer)
{
   if (isAlphabeticWithSpaces(textObj.value))
   {
      hideLayer(layer);
   } else
   {
      showLayer(layer);
   }
}


// Hide layer that contains the error image if textObj.value is a credit card number.
// Otherwise, show layer. 

function flagNonCreditCardNum(textObj, layer)
{
   if (isCreditCard(textObj.value) && textObj.value.length>=14 && textObj.value.length<=16) 
   {
      hideLayer(layer);
   } else 
   {
      showLayer(layer);
   }
}


// Hide layer that contains the error image if textObj.value is alphanumeric.
// Otherwise, show layer. 

function flagNonAlphanumeric(textObj, layer, requiredSize)
{
   var validInput;

   if (flagNonAlphanumeric.arguments.length == 3) // If a required size was entered
   {
      validInput = (isAlphanumeric(textObj.value) && textObj.value.length == requiredSize);
   } else
   {
      validInput = isAlphanumeric(textObj.value);
   }
      
   if (validInput)
   {
      hideLayer(layer);
   } else
   {
      showLayer(layer);
   }
}


// Hide layer that contains the error image if textObj.value is a date.
// Otherwise, show layer. 

function flagNonAOLDate(textObj, layer)
{
   if (isAOLDate(textObj))
   {
      hideLayer(layer);
   } else
   {
      showLayer(layer);
   }
}


// Hide layer that contains the error image if textObj.value is a credit card date.
// Otherwise, show layer. 

function flagNonAOLCCDate(textObj, layer, nonFormattedInput)
{
   if (isAOLCCDate(textObj,nonFormattedInput))
   {
      hideLayer(layer);
   } else
   {
      showLayer(layer);
   }
}


// Hide layer that contains the error image if textObj.value is an amount.
// Otherwise, show layer. 

function flagNonAOLAmount(textObj, layer, nonFormattedInput, maxLength)
{
   if (isAOLAmount(textObj, nonFormattedInput, false, maxLength))
   {
      hideLayer(layer);
   } else
   {
      showLayer(layer);
   }
}


// Hide layer that contains the error image if textObj.value is a time.
// Otherwise, show layer. 

function flagNonAOLTime(textObj, layer)
{
   if (isAOLTime(textObj))
   {
      hideLayer(layer);
   } else
   {
      showLayer(layer);
   }
}




// Hide layer that contains the error image if textObj.value is an integer.
// Otherwise, show layer. 

function flagNonInteger(textObj, layer, emptyOkay)
{
   var integerValue;
   if (flagNonInteger.arguments.length == 3) // If a value was entered for emptyOkay
   {
      integerValue = isInteger(textObj.value,emptyOkay);
   } else
   {
      integerValue = isInteger(textObj.value);
   }
   if (integerValue)
   {
      hideLayer(layer);
   } else
   {
      showLayer(layer);
   }
}


// Goes through the elements in thisForm to find inputs that have a "flag" function for 
// their onblur event. For each such input, checks to see whether the value is valid.
// If it is not, flag it as an error. At the end, if any errors have been found, alert 
// the user and return false. If there are no errors, return true.

function validateInput(thisForm)
{
   if (cancelButton == true)
   {
      return true;
   }

   var allowSubmit = true;
   var hasThirdParam = false; 
   var onBlurString, onBlurFunction, fIndex, onBlurDiv, firstBrace, firstParen, firstQuote, secondQuote, lastParen, comma;
   var numberOfEdits = 0;
   var editValues = new Array(thisForm.length); // An array as big as all form elements

   for (var i=0; i<thisForm.length; i++) // For every element in the form
   {    
      if (thisForm[i].tagName=="INPUT" && thisForm[i].onblur!=null
        && thisForm[i].disabled!=true) // If is input, has onblur event, and is not disabled
      {
         onBlurString = new String(thisForm[i].onblur); // Gets the value of onblur 
         firstBrace = onBlurString.indexOf('{'); // Finds the first brace
         fIndex = onBlurString.indexOf('f',firstBrace); // Finds location of f after brace
         firstParen = onBlurString.indexOf('(',firstBrace); // Finds location of (
         if (fIndex >= 0 && firstParen >= 0) // If there is an f and a ( in the string
         {
            if (onBlurString.substring(fIndex,fIndex+4) == "flag") // If "flag" exists
            {
               onBlurFunction = 
                 onBlurString.substring(fIndex,firstParen); // Gets full flag function name
               onBlurFunction = stripWhitespace(onBlurFunction); // Gets rid of any spaces
               firstQuote = onBlurString.indexOf("'",firstParen); // Location of ' in params
               secondQuote = onBlurString.indexOf("'",firstQuote+1); // Second ' in params
               if (firstQuote>=0 && secondQuote>=0) // If both quotes exist
               {
                  onBlurDiv = 
                    onBlurString.substring(firstQuote+1,secondQuote); // Value inside quotes
                  comma = onBlurString.indexOf(",",secondQuote); // Location of ,
                  if (comma>=0) // If there is a , (i.e., if there is a third parameter)
                  {
                     lastParen = onBlurString.indexOf(")",comma); // Location of )
                     hasThirdParam = true;
                     var thirdParam = onBlurString.substring(comma+1,lastParen);
		     thirdParam = stripCharsInBag(thirdParam," '");
                  }
                  switch(onBlurFunction) 
                  // In all cases, call flagErrorsOnSubmit and set the editValues array
                  // value to its result (true or false). Then increase the numberOfEdits.
                 
                  {
                     case "flagEmptyField":
                        editValues[numberOfEdits] = 
                         flagErrorsOnSubmit(isCompletedField(thisForm[i].value), onBlurDiv);
                        numberOfEdits++;
                        break;

                     case "flagNonText":
                        editValues[numberOfEdits] = 
                         flagErrorsOnSubmit(isAlphabeticWithSpaces(thisForm[i].value),
                           onBlurDiv);
                        numberOfEdits++;
                        break;

                     case "flagNonZipCode":
                        editValues[numberOfEdits] =
                         flagErrorsOnSubmit(isValidPostalCode(thisForm[i].value),onBlurDiv);
                        numberOfEdits++;
                        break;

                     case "flagNonAOLPhoneNumber":
                        editValues[numberOfEdits] =
                          flagErrorsOnSubmit(isAOLPhoneNumber(thisForm[i],thirdParam),
                            onBlurDiv);
                        numberOfEdits++;
                        break;
  
                     case "flagNonAlphanumeric":
                        if (hasThirdParam)
                        {
                           var inputValue = thisForm[i].value;
                           editValues[numberOfEdits] = 
                             flagErrorsOnSubmit((isAlphanumericWithSpaces(inputValue) &&
                               inputValue.length==thirdParam),onBlurDiv); 
                        } else
                        {
                           editValues[numberOfEdits] =
                             flagErrorsOnSubmit(isAlphanumericWithSpaces(thisForm[i].value), 
                               onBlurDiv);
                        }                        
                        numberOfEdits++;
                        break;

                     case "flagNonInteger":
                        if (hasThirdParam)
                        {
                           editValues[numberOfEdits] =
                            flagErrorsOnSubmit(
                              isInteger(thisForm[i].value,(thirdParam=="true")),onBlurDiv);
                        } else
                        {
                           editValues[numberOfEdits] =
                            flagErrorsOnSubmit(isInteger(thisForm[i].value),onBlurDiv);
                        }
                        numberOfEdits++;
                        break;

                     case "flagNonAOLDate":
                        editValues[numberOfEdits] =
                          flagErrorsOnSubmit(isAOLDate(thisForm[i]),onBlurDiv);
                        numberOfEdits++;
                        break;

                     case "flagNonAOLCCDate":
                        editValues[numberOfEdits] =
                          flagErrorsOnSubmit(isAOLCCDate(thisForm[i],thirdParam),onBlurDiv);
                        numberOfEdits++;
                        break;

                     case "flagNonAOLAmount":
                        editValues[numberOfEdits] =
                          flagErrorsOnSubmit(isAOLAmount(thisForm[i],thirdParam,false),onBlurDiv);
                        numberOfEdits++;
                        break;

                     case "flagNonAOLTime":
                        editValues[numberOfEdits] =
                          flagErrorsOnSubmit(isAOLTime(thisForm[i]),onBlurDiv);
                        numberOfEdits++;
                        break;

                     case "flagNonCreditCardNum":
                        editValues[numberOfEdits] =
                          flagErrorsOnSubmit((isCreditCard(thisForm[i].value)
                            && thisForm[i].value.length>=14 && thisForm[i].value.length<=16),onBlurDiv);
                        numberOfEdits++;
                        break;

                  }
               }
            }
         }
      }
   }
   for (var j=0; j<numberOfEdits; j++) // For each input that was validated
   {
      if (editValues[j] == false) // If the return value was false 
      {
         allowSubmit = false;
         break;
      }
   }
   if (allowSubmit == false) // If any return value was false;
   {
      alert("Please correct the flagged errors");
   }
   return allowSubmit; // Return false if there is an error; true if there is no error
}  


// Formats s as a phone number and inserts it as the value of textObj

function formatAOLPhoneNumber(s,textObj)
{
   s = stripCharsInBag(s,phoneNumberDelimiters); // Strip all characters but the numbers
   if (isInteger(s) && s.length == 10)
   {
      textObj.value = "("+s.substring(0,3)+") "+s.substring(3,6)+"-"+s.substring(6,10);
   }
}


// Takes the value from formattedTextObj and strips excess chars. If it is a valid
// phone number, set the value of formattedTextObj as the phone number and the value
// of nonFormattedInputName as just the digits in the phone number, and return true.
// If it is not a valid phone number, return false.

function isAOLPhoneNumber(formattedTextObj, nonFormattedInputName)
{
   var phoneNumber = formattedTextObj.value;
   phoneNumber = stripCharsInBag(phoneNumber, phoneNumberDelimiters);
   if (isInteger(phoneNumber) && phoneNumber.length == 10)
   {
      document.all(nonFormattedInputName).value = phoneNumber;
      formattedTextObj.value = "("+phoneNumber.substring(0,3)+") "+phoneNumber.substring(3,6)+"-"+phoneNumber.substring(6,10);
      return true;
   } else
   {
      return false;
   }
}


// Hide layer that contains the error image if textObj.value is a phone number.
// Otherwise, show layer. 

function flagNonAOLPhoneNumber(formattedTextObj, layer, nonFormattedInputName)
{
   if (isAOLPhoneNumber(formattedTextObj,nonFormattedInputName))
   {
      hideLayer(layer);
   } else
   {
      showLayer(layer);
   }
}


// Returns true either if cancelButton is true (usually when a cancel or back button has
// been clicked) or if at least one row has been selected from the PegaList with an ID
// that matches the string listID.

function validateListSelection(listID)
{
   if (cancelButton == true)
   {
      return true;
   } else
   {
      var numSelected = document.all(listID).NumSelectedRows();
      if (numSelected >= 1)
      {
         return true;
      } else
      {
         alert("Please select a value from the list");
         return false;
      }
   }
}

function limitTextArea(myArea, myLength) 
{	
   if(myArea.value.length >= myLength) 
   {
      alert("The textarea is full.");
      event.keyCode = 0;
   }
}

function checkTextAreaLength(myArea, myLength)
{
   if (myArea.value.length > myLength)
   {
      myArea.value = myArea.value.substring(0, myLength);
      alert("The textarea was too full. The data in the textarea had to be cut off.");
   }     
}
